/*!
 * HandwriterStylo SVG Stroke v0.1.0
 * "Pen-like" writing using real font glyph outlines (opentype.js) -> SVG paths.
 *
 * How it works:
 * - Loads a font file (TTF/OTF/WOFF; NOT WOFF2) via opentype.js (must be included separately).
 * - Lays out text into blocks/lines with word wrapping.
 * - Converts each glyph into an SVG filled path.
 * - Reveals the filled glyph via a thick stroked mask that is animated along the glyph outline path.
 * - Optional moving "pen tip" dot.
 *
 * This is much closer to "following the letter path" than a simple horizontal reveal.
 *
 * Requirements:
 * - Include opentype.js on the page (global `opentype`), e.g.
 *   <script src="https://unpkg.com/opentype.js@latest/dist/opentype.min.js"></script>
 * - Provide a fontUrl pointing to a .ttf/.otf/.woff version of BrittanySignature.
 *   (opentype.js usually does NOT parse .woff2)
 *
 * MIT License
 */
(function (root, factory) {
  if (typeof module === "object" && module.exports) module.exports = factory();
  else root.HandwriterStyloSVG = factory();
})(typeof self !== "undefined" ? self : this, function () {
  "use strict";

  const clamp = (n, a, b) => Math.max(a, Math.min(b, n));
  const rand = (a, b) => a + Math.random() * (b - a);
  const now = () => (typeof performance !== "undefined" ? performance.now() : Date.now());

  function prefersReducedMotion() {
    return typeof window !== "undefined"
      && window.matchMedia
      && window.matchMedia("(prefers-reduced-motion: reduce)").matches;
  }

  function sleep(ms, signal, pauseRef) {
    return new Promise((resolve, reject) => {
      if (signal?.aborted) return reject(Object.assign(new Error("aborted"), { name: "AbortError" }));
      const t0 = now();
      function tick() {
        if (signal?.aborted) return reject(Object.assign(new Error("aborted"), { name: "AbortError" }));
        if (pauseRef?.paused) return requestAnimationFrame(tick);
        const dt = now() - t0;
        if (dt >= ms) resolve();
        else requestAnimationFrame(tick);
      }
      requestAnimationFrame(tick);
    });
  }

  // Gentle continuous pen audio (optional)
  class PenAudio {
    constructor(opts = {}) {
      this.enabled = !!opts.enabled;
      this.volume = clamp(opts.volume ?? 0.16, 0, 1);
      this.roughness = clamp(opts.roughness ?? 0.50, 0, 1);
      this.baseLevel = clamp(opts.baseLevel ?? 0.08, 0, 1);
      this.transientLevel = clamp(opts.transientLevel ?? 0.08, 0, 1);
      this.transientEvery = Math.max(1, opts.transientEvery ?? 5);

      this._ctx = null;
      this._master = null;
      this._noiseSrc = null;
      this._baseGain = null;
      this._hitGain = null;
      this._bp = null;
      this._hp = null;
      this._panner = null;
      this._count = 0;
    }

    _audioContext() {
      const AudioCtx = window.AudioContext || window.webkitAudioContext;
      return AudioCtx ? new AudioCtx() : null;
    }

    _makeNoiseBuffer(ctx) {
      const seconds = 1.0;
      const buffer = ctx.createBuffer(1, Math.floor(ctx.sampleRate * seconds), ctx.sampleRate);
      const data = buffer.getChannelData(0);
      for (let i = 0; i < data.length; i++) data[i] = (Math.random() * 2 - 1);
      return buffer;
    }

    async _ensure() {
      if (!this.enabled) return;
      if (!this._ctx) {
        this._ctx = this._audioContext();
        if (!this._ctx) return;

        this._master = this._ctx.createGain();
        this._master.gain.value = this.volume;
        this._master.connect(this._ctx.destination);

        const noiseBuf = this._makeNoiseBuffer(this._ctx);
        this._noiseSrc = this._ctx.createBufferSource();
        this._noiseSrc.buffer = noiseBuf;
        this._noiseSrc.loop = true;

        this._baseGain = this._ctx.createGain();
        this._baseGain.gain.value = 0;

        this._hitGain = this._ctx.createGain();
        this._hitGain.gain.value = 0;

        this._hp = this._ctx.createBiquadFilter();
        this._hp.type = "highpass";
        this._hp.frequency.value = 180;

        this._bp = this._ctx.createBiquadFilter();
        this._bp.type = "bandpass";
        this._bp.frequency.value = 1400;
        this._bp.Q.value = 1.2;

        this._panner = this._ctx.createStereoPanner();
        this._panner.pan.value = 0;

        this._noiseSrc.connect(this._baseGain);
        this._noiseSrc.connect(this._hitGain);

        const merger = this._ctx.createGain();
        this._baseGain.connect(merger);
        this._hitGain.connect(merger);

        merger.connect(this._hp);
        this._hp.connect(this._bp);
        this._bp.connect(this._panner);
        this._panner.connect(this._master);

        this._noiseSrc.start();
      }
    }

    async unlock() {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx) return;
      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}
    }

    autoUnlock(target = document) {
      if (!this.enabled) return () => {};
      const handler = async () => { await this.unlock(); cleanup(); };
      const cleanup = () => {
        target.removeEventListener("pointerdown", handler);
        target.removeEventListener("touchstart", handler);
        target.removeEventListener("keydown", handler);
      };
      target.addEventListener("pointerdown", handler, { once: true });
      target.addEventListener("touchstart", handler, { once: true });
      target.addEventListener("keydown", handler, { once: true });
      return cleanup;
    }

    async penDown(intensity = 1) {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx) return;
      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}

      const t = this._ctx.currentTime;
      const i = clamp(intensity, 0.2, 1);
      this._count++;

      this._panner.pan.setValueAtTime((Math.random() * 2 - 1) * 0.12, t);

      const baseFreq = clamp(1200 + rand(-220, 420) + this.roughness * rand(80, 420), 900, 2600);
      this._bp.frequency.setValueAtTime(baseFreq, t);
      this._bp.Q.setValueAtTime(clamp(1.0 + rand(-0.1, 0.6) + this.roughness * 0.45, 0.8, 2.1), t);

      const base = (this.baseLevel * 0.14) * i;
      this._baseGain.gain.cancelScheduledValues(t);
      this._baseGain.gain.setValueAtTime(this._baseGain.gain.value, t);
      this._baseGain.gain.linearRampToValueAtTime(base, t + 0.04);

      if (this.transientLevel > 0 && (this._count % this.transientEvery === 0)) {
        const hit = (this.transientLevel * 0.12) * i;
        this._hitGain.gain.cancelScheduledValues(t);
        this._hitGain.gain.setValueAtTime(this._hitGain.gain.value, t);
        this._hitGain.gain.linearRampToValueAtTime(hit, t + 0.02);
        this._hitGain.gain.linearRampToValueAtTime(0, t + 0.22);
      }
    }

    penUp() {
      if (!this.enabled || !this._ctx) return;
      const t = this._ctx.currentTime;
      this._baseGain.gain.cancelScheduledValues(t);
      this._baseGain.gain.setValueAtTime(this._baseGain.gain.value, t);
      this._baseGain.gain.linearRampToValueAtTime(0, t + 0.10);
      this._hitGain.gain.cancelScheduledValues(t);
      this._hitGain.gain.setValueAtTime(0, t);
    }
  }

  function svgEl(name) {
    return document.createElementNS("http://www.w3.org/2000/svg", name);
  }

  async function loadFont(fontUrl) {
    if (!window.opentype) throw new Error("opentype.js is required (global `opentype`).");
    const res = await fetch(fontUrl);
    const buf = await res.arrayBuffer();
    return window.opentype.parse(buf);
  }

  // Layout helpers (word wrap with font metrics)
  function fontScale(font, fontSizePx) {
    return fontSizePx / font.unitsPerEm;
  }

  function advanceWidth(font, leftGlyph, rightGlyph, fontSizePx) {
    const s = fontScale(font, fontSizePx);
    const kern = rightGlyph ? font.getKerningValue(leftGlyph, rightGlyph) : 0;
    return (leftGlyph.advanceWidth + kern) * s;
  }

  function measureTextWidth(font, text, fontSizePx) {
    const glyphs = font.stringToGlyphs(text);
    let w = 0;
    for (let i = 0; i < glyphs.length; i++) {
      const g = glyphs[i];
      const next = glyphs[i + 1];
      w += advanceWidth(font, g, next, fontSizePx);
    }
    return w;
  }

  function wrapText(font, text, fontSizePx, maxWidth) {
    // Preserve explicit newlines
    const paragraphs = String(text || "").split("\n");
    const lines = [];
    for (const para of paragraphs) {
      if (!para.trim()) {
        lines.push(""); // blank line
        continue;
      }
      const words = para.split(/\s+/);
      let line = "";
      for (let wi = 0; wi < words.length; wi++) {
        const w = words[wi];
        const candidate = line ? (line + " " + w) : w;
        const width = measureTextWidth(font, candidate, fontSizePx);
        if (width <= maxWidth || !line) {
          line = candidate;
        } else {
          lines.push(line);
          line = w;
        }
      }
      if (line) lines.push(line);
    }
    return lines;
  }

  class SVGWriter {
    constructor(container, opts = {}) {
      if (!container) throw new Error("SVGWriter needs a container element");
      this.el = container;
      this.opts = normalizeOptions(opts);

      this._abort = null;
      this._pause = { paused: false };
      this._autoPaused = false;

      this.audio = new PenAudio(this.opts.sound);
      if (this.opts.sound.autoUnlock) this.audio.autoUnlock(document);

      // Create svg root
      this.svg = svgEl("svg");
      this.svg.setAttribute("width", "100%");
      this.svg.setAttribute("height", "100%");
      this.svg.style.display = "block";
      this.svg.style.overflow = "visible";
      this.el.innerHTML = "";
      this.el.appendChild(this.svg);

      this.defs = svgEl("defs");
      this.svg.appendChild(this.defs);

      this.layer = svgEl("g");
      this.svg.appendChild(this.layer);

      // focus/visibility
      this._vis = () => {
        if (document.hidden) {
          if (this.opts.pauseWhenHidden && !this._pause.paused) { this._autoPaused = true; this.pause(); }
          this.audio.penUp();
        } else {
          if (this.opts.pauseWhenHidden && this._autoPaused) { this._autoPaused = false; this.resume(); }
        }
      };
      document.addEventListener("visibilitychange", this._vis);
      window.addEventListener("blur", this._vis);
      window.addEventListener("focus", this._vis);
    }

    pause() { this._pause.paused = true; }
    resume() { this._pause.paused = false; }

    stop() {
      if (this._abort) this._abort.abort();
      this._abort = null;
      this.audio.penUp();
    }

    clear() {
      this.stop();
      while (this.defs.firstChild) this.defs.removeChild(this.defs.firstChild);
      while (this.layer.firstChild) this.layer.removeChild(this.layer.firstChild);
    }

    async unlockAudio() { await this.audio.unlock(); }

    writeOnUserGesture(blocks, options = {}) {
      const once = async () => {
        document.removeEventListener("pointerdown", once);
        document.removeEventListener("touchstart", once);
        document.removeEventListener("keydown", once);
        await this.unlockAudio();
        await this.writeRich(blocks, options);
      };
      document.addEventListener("pointerdown", once, { once: true });
      document.addEventListener("touchstart", once, { once: true });
      document.addEventListener("keydown", once, { once: true });
    }

    async writeRich(blocks, options = {}) {
      const mode = options.mode || this.opts.mode;
      if (mode === "replace") this.clear();

      // Reduced motion shortcut
      const reduced = (this.opts.reducedMotion === true) || (this.opts.reducedMotion !== false && prefersReducedMotion());
      this._abort = new AbortController();
      const signal = this._abort.signal;

      if (!this.font) {
        this.font = await loadFont(this.opts.fontUrl);
      }

      // Layout all glyphs first, so everything is in final place (no shifting)
      const layout = this._layoutBlocks(this.font, blocks);
      this._buildSvg(layout);

      if (reduced) {
        // show all instantly
        for (const g of layout.glyphRuns) {
          g.fillPath.style.opacity = "1";
          g.fillPath.removeAttribute("mask");
        }
        return;
      }

      // Animate glyph-by-glyph with your same timing model
      for (const ev of layout.events) {
        if (signal.aborted) break;
        while (this._pause.paused && !signal.aborted) await new Promise(r => setTimeout(r, 50));
        if (signal.aborted) break;

        if (ev.type === "pause") {
          this.audio.penUp();
          await sleep(ev.ms, signal, this._pause);
          continue;
        }

        if (ev.type === "glyph") {
          await this._animateGlyph(ev, signal);
        }
      }

      this.audio.penUp();
    }

    _layoutBlocks(font, blocks) {
      const rect = this.el.getBoundingClientRect();
      const W = rect.width;
      const H = rect.height;

      const glyphRuns = [];
      const events = [];

      // First: compute lines with wrapping
      let totalHeight = 0;
      const linesAll = [];

      for (const b of blocks) {
        const fontSize = resolveFontSizePx(b.fontSize ?? this.opts.fontSize) * (b.size ?? 1);
        const lineHeight = fontSize * (b.lineHeight ?? this.opts.lineHeight);
        const mt = b.marginTop ?? 0;
        const mb = b.marginBottom ?? 0;

        const maxWidth = W; // writer area should already be inset by CSS
        const wrappedLines = wrapText(font, b.text ?? "", fontSize, maxWidth);

        for (const ln of wrappedLines) {
          linesAll.push({
            block: b,
            text: ln,
            fontSize,
            lineHeight,
            mt, mb
          });
          totalHeight += mt + lineHeight + mb;
        }
      }

      const startY = (this.opts.verticalAlign === "top") ? 0 : (H - totalHeight) / 2;
      let y = startY;

      // Second: for each line, create glyph runs (positioned)
      const punctuationSet = this.opts.punctuationSet;

      for (let li = 0; li < linesAll.length; li++) {
        const L = linesAll[li];
        const b = L.block;
        const fontSize = L.fontSize;
        const lineHeight = L.lineHeight;

        y += L.mt;

        const text = L.text;
        const lineWidth = measureTextWidth(font, text, fontSize);

        let x = 0;
        const align = b.align ?? this.opts.align;
        if (align === "center") x = (W - lineWidth) / 2;
        else if (align === "right") x = (W - lineWidth);

        const baselineY = y + fontSize; // approximate baseline

        // Timing overrides per block
        const minDelay = (b.minDelay ?? this.opts.minDelay) * (b.timeScale ?? this.opts.timeScale);
        const maxDelay = (b.maxDelay ?? this.opts.maxDelay) * (b.timeScale ?? this.opts.timeScale);
        const revealMin = (b.revealMin ?? this.opts.revealMin) * (b.timeScale ?? this.opts.timeScale);
        const revealMax = (b.revealMax ?? this.opts.revealMax) * (b.timeScale ?? this.opts.timeScale);
        const punctMin = (b.punctuationMin ?? this.opts.punctuationMin) * (b.timeScale ?? this.opts.timeScale);
        const punctMax = (b.punctuationMax ?? this.opts.punctuationMax) * (b.timeScale ?? this.opts.timeScale);
        const wordPauseMin = (b.wordPauseMin ?? this.opts.wordPauseMin) * (b.timeScale ?? this.opts.timeScale);
        const wordPauseMax = (b.wordPauseMax ?? this.opts.wordPauseMax) * (b.timeScale ?? this.opts.timeScale);
        const linePauseMin = (b.linePauseMin ?? this.opts.linePauseMin) * (b.timeScale ?? this.opts.timeScale);
        const linePauseMax = (b.linePauseMax ?? this.opts.linePauseMax) * (b.timeScale ?? this.opts.timeScale);

        // Build glyphs
        const glyphs = font.stringToGlyphs(text);
        let penX = x;

        for (let gi = 0; gi < glyphs.length; gi++) {
          const g = glyphs[gi];
          const next = glyphs[gi + 1];

          const ch = text[gi] ?? "";
          // Space: word pause
          if (ch === " " || ch === "\t") {
            const wp = rand(wordPauseMin, wordPauseMax);
            if (wp > 0) events.push({ type: "pause", ms: wp });
            penX += advanceWidth(font, g, next, fontSize);
            continue;
          }

          const path = g.getPath(penX, baselineY, fontSize);
          const d = path.toPathData(3);

          const run = {
            ch,
            d,
            fontSize,
            penWidth: this.opts.penWidth,
            inkColor: this.opts.inkColor,
            // timing per glyph
            dur: rand(revealMin, revealMax),
          };
          glyphRuns.push(run);

          // schedule
          events.push({ type: "glyph", run });

          // punctuation pause or letter gap
          if (this.opts.punctuationPauses && punctuationSet.has(ch)) {
            const pp = rand(punctMin, punctMax);
            if (pp > 0) events.push({ type: "pause", ms: pp });
          } else {
            const gap = rand(minDelay, maxDelay);
            if (gap > 0) events.push({ type: "pause", ms: gap });
          }

          penX += advanceWidth(font, g, next, fontSize);
        }

        // end-of-line pause
        const lp = rand(linePauseMin, linePauseMax);
        if (lp > 0) events.push({ type: "pause", ms: lp });

        y += lineHeight + L.mb;
      }

      return { glyphRuns, events };
    }

    _buildSvg(layout) {
      // Build paths in final positions (hidden), plus masks for each glyph
      const ink = this.opts.inkColor;
      const penW = this.opts.penWidth;

      // Clear previous
      while (this.defs.firstChild) this.defs.removeChild(this.defs.firstChild);
      while (this.layer.firstChild) this.layer.removeChild(this.layer.firstChild);

      layout.glyphRuns.forEach((run, idx) => {
        const maskId = `hwMask_${idx}_${Math.floor(Math.random()*1e9)}`;

        const mask = svgEl("mask");
        mask.setAttribute("id", maskId);

        // mask background black (hidden)
        const bg = svgEl("rect");
        bg.setAttribute("x", "-5000");
        bg.setAttribute("y", "-5000");
        bg.setAttribute("width", "10000");
        bg.setAttribute("height", "10000");
        bg.setAttribute("fill", "black");
        mask.appendChild(bg);

        // this is the "pen path" that reveals the fill
        const stroke = svgEl("path");
        stroke.setAttribute("d", run.d);
        stroke.setAttribute("fill", "none");
        stroke.setAttribute("stroke", "white");
        stroke.setAttribute("stroke-linecap", "round");
        stroke.setAttribute("stroke-linejoin", "round");
        stroke.setAttribute("stroke-width", String(penW));
        mask.appendChild(stroke);

        this.defs.appendChild(mask);

        // Filled glyph (masked)
        const fillPath = svgEl("path");
        fillPath.setAttribute("d", run.d);
        fillPath.setAttribute("fill", ink);
        fillPath.setAttribute("opacity", "1");
        fillPath.setAttribute("mask", `url(#${maskId})`);

        // optional subtle ink shadow (very light)
        if (this.opts.inkShadow) {
          fillPath.style.filter = "drop-shadow(0.3px 0.5px 0.6px rgba(0,0,0,0.22))";
        }

        this.layer.appendChild(fillPath);

        // Save references for animation
        run.maskId = maskId;
        run.strokeEl = stroke;
        run.fillPath = fillPath;

        // Prepare dash animation
        try {
          const len = stroke.getTotalLength();
          run.len = len;
          stroke.style.strokeDasharray = String(len);
          stroke.style.strokeDashoffset = String(len);
        } catch (_) {
          run.len = 0;
        }

        // pen tip dot
        if (this.opts.showPenTip) {
          const dot = svgEl("circle");
          dot.setAttribute("r", String(penW * 0.45));
          dot.setAttribute("fill", ink);
          dot.style.opacity = "0";
          this.layer.appendChild(dot);
          run.dotEl = dot;
        }
      });
    }

    async _animateGlyph(ev, signal) {
      const run = ev.run;
      const stroke = run.strokeEl;
      const dot = run.dotEl;

      const len = run.len || 0;
      if (!len || !stroke) return;

      const t0 = now();
      const dur = Math.max(40, run.dur);

      // Sound intensity: bigger glyphs -> slightly more
      await this.audio.penDown(0.75);

      if (dot) dot.style.opacity = "0.9";

      while (true) {
        if (signal.aborted) break;
        while (this._pause.paused && !signal.aborted) await new Promise(r => setTimeout(r, 50));
        if (signal.aborted) break;

        const t = now();
        const p = clamp((t - t0) / dur, 0, 1);

        const off = (1 - p) * len;
        stroke.style.strokeDashoffset = String(off);

        if (dot && typeof stroke.getPointAtLength === "function") {
          const pt = stroke.getPointAtLength(Math.max(0, Math.min(len, len - off)));
          dot.setAttribute("cx", String(pt.x));
          dot.setAttribute("cy", String(pt.y));
        }

        if (p >= 1) break;
        await new Promise(r => requestAnimationFrame(r));
      }

      // Make sure fully revealed
      stroke.style.strokeDashoffset = "0";
      if (dot) dot.style.opacity = "0";
    }
  }

  function resolveFontSizePx(v) {
    if (typeof v === "number") return v;
    const s = String(v || "").trim();
    // Accept clamp(...) but pick the middle value as approximate for layout
    if (s.startsWith("clamp(")) {
      const m = s.match(/clamp\(\s*([0-9.]+)px\s*,\s*([0-9.]+)vw\s*,\s*([0-9.]+)px\s*\)/i);
      if (m) return parseFloat(m[1]) + (parseFloat(m[3]) - parseFloat(m[1])) * 0.5;
      return 44;
    }
    const px = s.match(/([0-9.]+)\s*px/i);
    if (px) return parseFloat(px[1]);
    const n = parseFloat(s);
    return Number.isFinite(n) ? n : 44;
  }

  function normalizeOptions(o) {
    const punctChars = o.punctuationChars || ".,;:!?";
    return {
      mode: o.mode || "replace",

      // IMPORTANT: For opentype.js, use a TTF/OTF/WOFF font file (not WOFF2).
      fontUrl: o.fontUrl || "",
      fontSize: o.fontSize || "42px",
      lineHeight: o.lineHeight ?? 1.22,
      align: o.align || "center",
      verticalAlign: o.verticalAlign || "center",

      inkColor: o.inkColor || "rgba(0,0,0,0.92)",
      inkShadow: o.inkShadow ?? true,

      penWidth: o.penWidth ?? 2.6,
      showPenTip: o.showPenTip ?? false,

      // timing defaults
      minDelay: o.minDelay ?? 35,
      maxDelay: o.maxDelay ?? 90,
      revealMin: o.revealMin ?? 160,
      revealMax: o.revealMax ?? 300,
      timeScale: o.timeScale ?? 1.0,

      punctuationPauses: o.punctuationPauses ?? true,
      punctuationMin: o.punctuationMin ?? 180,
      punctuationMax: o.punctuationMax ?? 650,
      punctuationSet: new Set(punctChars.split("")),

      wordPauseMin: o.wordPauseMin ?? 80,
      wordPauseMax: o.wordPauseMax ?? 220,

      linePauseMin: o.linePauseMin ?? 0,
      linePauseMax: o.linePauseMax ?? 0,

      sound: {
        enabled: o.sound?.enabled ?? true,
        autoUnlock: o.sound?.autoUnlock ?? true,
        volume: o.sound?.volume ?? 0.16,
        roughness: o.sound?.roughness ?? 0.50,
        baseLevel: o.sound?.baseLevel ?? 0.08,
        transientLevel: o.sound?.transientLevel ?? 0.08,
        transientEvery: o.sound?.transientEvery ?? 5,
      },

      reducedMotion: o.reducedMotion ?? null,
      pauseWhenHidden: o.pauseWhenHidden ?? true,
    };
  }

  function create(container, options) {
    return new SVGWriter(container, options);
  }

  return { create };
});
