/*!
 * HandwriterStylo PRO v0.3.20
 * Fixes:
 * 1) Word wrapping: letters are grouped into .hw-word (inline-block) so words don't break mid-word.
 * 2) Descender clipping: expand clip-path vertically + add vertical padding (without affecting layout too much).
 */
(function (root, factory) {
  if (typeof module === "object" && module.exports) module.exports = factory();
  else root.HandwriterStylo = factory();
})(typeof self !== "undefined" ? self : this, function () {
  "use strict";

  const STYLE_ID = "handwriter-stylo-pro-style";

  const clamp = (n, a, b) => Math.max(a, Math.min(b, n));
  const rand = (a, b) => a + Math.random() * (b - a);

  function prefersReducedMotion() {
    return typeof window !== "undefined"
      && window.matchMedia
      && window.matchMedia("(prefers-reduced-motion: reduce)").matches;
  }

  function injectStylesOnce() {
    if (typeof document === "undefined") return;
    if (document.getElementById(STYLE_ID)) return;

    const style = document.createElement("style");
    style.id = STYLE_ID;
    style.textContent = `
      .hw-stylo {
        display: block;
        width: 100%;
        max-width: 100%;
        min-width: 0;

        white-space: pre-wrap;
        line-height: var(--hw-line-height, 1.18);
        letter-spacing: 0.01em;
        font-kerning: normal;
        font-feature-settings: "liga" 1, "calt" 1;
        -webkit-font-smoothing: antialiased;
        -moz-osx-font-smoothing: grayscale;
        text-rendering: geometricPrecision;
        color: var(--hw-ink, rgba(12,12,12,0.82));

        overflow: visible; /* do not clip swashes/descenders */
      }

      .hw-stylo.hw-ink {
        text-shadow:
          0.35px 0.45px 0.8px rgba(0,0,0,0.18),
          0px 0.45px 1.3px rgba(0,0,0,0.10);
      }

      .hw-line {
        display: block;
        width: 100%;
        max-width: 100%;
        min-width: 0;
        flex: 0 0 auto;
        overflow: visible;
      }

      /* Prevent mid-word breaks */
      .hw-word {
        display: inline-block;
        white-space: nowrap;
        overflow: visible;
      }

      .hw-char {
        display: inline-block;
        opacity: 0;
        transform-origin: 50% 78%;
        will-change: clip-path, opacity, transform, filter;

        /* Horizontal padding for left/right swashes */
        padding-left: var(--hw-char-pad-x, 0.16em);
        padding-right: var(--hw-char-pad-x, 0.16em);
        margin-left: calc(-1 * var(--hw-char-pad-x, 0.16em));
        margin-right: calc(-1 * var(--hw-char-pad-x, 0.16em));

        /* Vertical padding for descenders (does NOT affect wrapping) */
        padding-top: var(--hw-char-pad-y, 0.10em);
        padding-bottom: var(--hw-char-pad-y, 0.18em);
        margin-top: calc(-1 * var(--hw-char-pad-y, 0.10em));
        margin-bottom: calc(-1 * var(--hw-char-pad-y, 0.18em));

        overflow: visible;
      }

      .hw-char.hw-on {
        animation: hw-fade var(--hw-fade-ms, 90ms) linear forwards;
      }

      @supports (clip-path: inset(0 100% 0 0)) {
        .hw-char {
          /* Expand clip to avoid chopping swashes/descenders */
          clip-path: inset(var(--hw-clip-pad-y, -0.30em) 100% var(--hw-clip-pad-y, -0.30em) var(--hw-clip-pad-x, -0.40em));
          filter: blur(0.18px);
        }
        .hw-char.hw-on {
          animation: hw-reveal var(--hw-reveal-ms, 120ms) linear forwards;
        }
      }

      @keyframes hw-reveal {
        to { clip-path: inset(var(--hw-clip-pad-y, -0.30em) 0 var(--hw-clip-pad-y, -0.30em) var(--hw-clip-pad-x, -0.40em));
             opacity: var(--hw-op, 1); filter: blur(0px); }
      }
      @keyframes hw-fade {
        to { opacity: var(--hw-op, 1); }
      }

      /* Pen-like reveal: diagonal moving front (set revealMode:"pen") */
      @supports (clip-path: polygon(0 0, 100% 0, 100% 100%, 0 100%)) {
        /* IMPORTANT: do NOT animate all chars immediately.
           We only animate when .hw-on is added, so lockLayoutX doesn't make a whole line appear at once. */
        .hw-stylo.hw-pen .hw-char {
          position: relative;
          overflow: visible;
          filter: blur(0.14px);
          /* start fully hidden */
          clip-path: polygon(-10% 0, -10% 0, -10% 100%, -10% 100%);
        }

        /* A tiny "wet ink" zone that travels with the pen front.
           It's subtle (not a cursor), but makes letters feel written now. */
        .hw-stylo.hw-pen .hw-char::after {
          content: "";
          position: absolute;
          inset: -0.35em -0.25em;
          pointer-events: none;
          opacity: 0;
          mix-blend-mode: multiply;
          background:
            radial-gradient(circle at 20% 60%,
              rgba(0,0,0,0.20),
              rgba(0,0,0,0.10) 25%,
              rgba(0,0,0,0.00) 60%);
          transform: translateX(-25%);
        }

        .hw-stylo.hw-pen .hw-char.hw-on {
          animation: hw-pen-clip-down-bot var(--hw-reveal-ms, 140ms) linear forwards;
        }
        .hw-stylo.hw-pen .hw-char.hw-on::after {
          animation: hw-pen-tip-down-bot var(--hw-reveal-ms, 140ms) linear forwards;
        }

        /* Top-first variation */
        .hw-stylo.hw-pen .hw-char.hw-on.hw-pen-top {
          animation: hw-pen-clip-down-top var(--hw-reveal-ms, 140ms) linear forwards;
        }
        .hw-stylo.hw-pen .hw-char.hw-on.hw-pen-top::after {
          animation: hw-pen-tip-down-top var(--hw-reveal-ms, 140ms) linear forwards;
        }

        /* Opposite diagonal */
        .hw-stylo.hw-pen .hw-char.hw-on.hw-pen-up {
          animation: hw-pen-clip-up-bot var(--hw-reveal-ms, 140ms) linear forwards;
        }
        .hw-stylo.hw-pen .hw-char.hw-on.hw-pen-up::after {
          animation: hw-pen-tip-up-bot var(--hw-reveal-ms, 140ms) linear forwards;
        }

        /* Opposite diagonal + top-first */
        .hw-stylo.hw-pen .hw-char.hw-on.hw-pen-up.hw-pen-top {
          animation: hw-pen-clip-up-top var(--hw-reveal-ms, 140ms) linear forwards;
        }
        .hw-stylo.hw-pen .hw-char.hw-on.hw-pen-up.hw-pen-top::after {
          animation: hw-pen-tip-up-top var(--hw-reveal-ms, 140ms) linear forwards;
        }

        @keyframes hw-pen-clip-down-bot {
          0% {
            opacity: 0;
            clip-path: polygon(-10% 0, -10% 0, -10% 100%, -10% 100%);
          }
          10% { opacity: 0.25; }
          24% { opacity: var(--hw-op, 1); }

          /* First: a low band (bottom-up feel) */
          52% { clip-path: polygon(-10% 55%, 62% 42%, 90% 100%, -10% 100%); }

          /* Then: expand to full-height diagonal */
          72% { clip-path: polygon(-10% 0, 70% 0, 95% 100%, -10% 100%); }

          100% {
            opacity: var(--hw-op, 1);
            filter: blur(0px);
            clip-path: polygon(-10% 0, 110% 0, 110% 100%, -10% 100%);
          }
        }

        @keyframes hw-pen-clip-down-top {
          0% {
            opacity: 0;
            clip-path: polygon(-10% 0, -10% 0, -10% 100%, -10% 100%);
          }
          10% { opacity: 0.25; }
          24% { opacity: var(--hw-op, 1); }

          /* First: a high band (top-down feel) */
          52% { clip-path: polygon(-10% 0, 90% 0, 62% 60%, -10% 45%); }

          72% { clip-path: polygon(-10% 0, 70% 0, 95% 100%, -10% 100%); }

          100% {
            opacity: var(--hw-op, 1);
            filter: blur(0px);
            clip-path: polygon(-10% 0, 110% 0, 110% 100%, -10% 100%);
          }
        }

        /* Opposite diagonal (variation) */
        @keyframes hw-pen-clip-up-bot {
          0% {
            opacity: 0;
            clip-path: polygon(-10% 100%, -10% 100%, -10% 0, -10% 0);
          }
          10% { opacity: 0.25; }
          24% { opacity: var(--hw-op, 1); }

          52% { clip-path: polygon(-10% 55%, 90% 100%, 62% 42%, -10% 100%); }
          72% { clip-path: polygon(-10% 100%, 70% 100%, 95% 0, -10% 0); }

          100% {
            opacity: var(--hw-op, 1);
            filter: blur(0px);
            clip-path: polygon(-10% 100%, 110% 100%, 110% 0, -10% 0);
          }
        }

        @keyframes hw-pen-clip-up-top {
          0% {
            opacity: 0;
            clip-path: polygon(-10% 100%, -10% 100%, -10% 0, -10% 0);
          }
          10% { opacity: 0.25; }
          24% { opacity: var(--hw-op, 1); }

          52% { clip-path: polygon(-10% 0, 62% 60%, 90% 0, -10% 45%); }
          72% { clip-path: polygon(-10% 100%, 70% 100%, 95% 0, -10% 0); }

          100% {
            opacity: var(--hw-op, 1);
            filter: blur(0px);
            clip-path: polygon(-10% 100%, 110% 100%, 110% 0, -10% 0);
          }
        }

        /* Wet-ink zone moves with both X and Y for more "just written" feel */
        @keyframes hw-pen-tip-down-bot {
          0% { opacity: 0; transform: translate(-25%, 10%); }
          12% { opacity: 0.35; }
          72% { opacity: 0.18; transform: translate(78%, 16%); }
          100% { opacity: 0; transform: translate(110%, 10%); }
        }
        @keyframes hw-pen-tip-down-top {
          0% { opacity: 0; transform: translate(-25%, -8%); }
          12% { opacity: 0.35; }
          72% { opacity: 0.18; transform: translate(78%, -12%); }
          100% { opacity: 0; transform: translate(110%, -8%); }
        }
        @keyframes hw-pen-tip-up-bot {
          0% { opacity: 0; transform: translate(-25%, 10%); }
          12% { opacity: 0.35; }
          72% { opacity: 0.18; transform: translate(78%, 16%); }
          100% { opacity: 0; transform: translate(110%, 10%); }
        }
        @keyframes hw-pen-tip-up-top {
          0% { opacity: 0; transform: translate(-25%, -8%); }
          12% { opacity: 0.35; }
          72% { opacity: 0.18; transform: translate(78%, -12%); }
          100% { opacity: 0; transform: translate(110%, -8%); }
        }
          10%  { opacity: 0.55; }
          70%  { opacity: 0.35; transform: translateX(55%); }
          100% { opacity: 0; transform: translateX(95%); }
        }
          12% { opacity: 0.25; }
          28% { opacity: var(--hw-op, 1); }
          70% { clip-path: polygon(-10% 100%, 70% 100%, 95% 0, -10% 0); }
          100% {
            opacity: var(--hw-op, 1);
            filter: blur(0px);
            clip-path: polygon(-10% 100%, 110% 100%, 110% 0, -10% 0);
          }
        }
      }
      }

      /* If you still see clipping/* If you still see clipping with some signature fonts, use revealMode:"opacity" */
      .hw-stylo.hw-opacity .hw-char {
        clip-path: none !important;
        filter: none !important;
      }
      .hw-stylo.hw-opacity .hw-char.hw-on {
        animation: hw-fade var(--hw-fade-ms, 90ms) linear forwards !important;
      }

      .hw-cursor {
        display: inline-block;
        width: 0.55ch;
        transform: translateY(0.08em);
        opacity: 0.55;
        animation: hw-blink 0.9s steps(1) infinite;
        user-select: none;
        pointer-events: none;
      }
      @keyframes hw-blink { 50% { opacity: 0; } }
    `;
    document.head.appendChild(style);
  }

  async function wait(ms, signal, pauseState) {
    const started = (typeof performance !== "undefined" ? performance.now() : Date.now());
    let target = started + ms;

    while (true) {
      if (signal?.aborted) throw Object.assign(new Error("aborted"), { name: "AbortError" });

      if (pauseState?.paused) {
        await new Promise(r => setTimeout(r, 50));
        const now = (typeof performance !== "undefined" ? performance.now() : Date.now());
        target = now + Math.max(0, target - now);
        continue;
      }

      const now = (typeof performance !== "undefined" ? performance.now() : Date.now());
      const remaining = target - now;
      if (remaining <= 0) return;
      await new Promise(r => setTimeout(r, Math.min(50, remaining)));
    }
  }

  function nextPaintFrame() {
    return new Promise((resolve) => {
      if (typeof requestAnimationFrame === "function") requestAnimationFrame(() => resolve());
      else setTimeout(resolve, 0);
    });
  }

  class PenAudio {
    constructor(opts = {}) {
      this.enabled = !!opts.enabled;
      this.volume = clamp(opts.volume ?? 0.16, 0, 1);

      // Character (0..1)
      this.roughness = clamp(opts.roughness ?? 0.55, 0, 1);     // lower = smoother
      this.squeakChance = clamp(opts.squeakChance ?? 0.03, 0, 1);
      this.stereo = clamp(opts.stereo ?? 0.14, 0, 1);

      // More continuous handwriting (not typewriter):
      this.baseLevel = clamp(opts.baseLevel ?? 0.08, 0, 1);      // continuous friction while moving
      this.transientLevel = clamp(opts.transientLevel ?? 0.10, 0, 1); // small accents only
      this.transientEvery = Math.max(1, (opts.transientEvery ?? 4));  // accent every N letters

      // Envelope smoothing (seconds)
      this.baseAttack = clamp(opts.baseAttack ?? 0.035, 0.001, 0.2);
      this.baseRelease = clamp(opts.baseRelease ?? 0.10, 0.005, 0.5);
      this.hitAttack = clamp(opts.hitAttack ?? 0.020, 0.001, 0.2);
      this.hitReleaseMin = clamp(opts.hitReleaseMin ?? 0.18, 0.01, 1.0);
      this.hitReleaseMax = clamp(opts.hitReleaseMax ?? 0.26, 0.01, 1.0);

      // Suspending can break re-audio on some browsers; keep optional
      this.allowSuspend = (opts.allowSuspend ?? false);

      this._ctx = null;
      this._master = null;

      this._noiseSrc = null;
      this._baseGain = null;
      this._hitGain = null;

      this._bp = null;
      this._hp = null;
      this._sat = null;
      this._comp = null;
      this._panner = null;

      this._lfo = null;
      this._lfoGain = null;

      // continuity helpers
      this._lastT = 0;
      this._lastFreq = 1500;
      this._lastQ = 1.5;
      this._charCount = 0;
    }

    _audioContext() {
      const AudioCtx = window.AudioContext || window.webkitAudioContext;
      return AudioCtx ? new AudioCtx() : null;
    }

    _makeNoiseBuffer(ctx) {
      const seconds = 1.0;
      const buffer = ctx.createBuffer(1, Math.floor(ctx.sampleRate * seconds), ctx.sampleRate);
      const data = buffer.getChannelData(0);
      for (let i = 0; i < data.length; i++) data[i] = (Math.random() * 2 - 1);
      return buffer;
    }

    _makeSaturator(ctx, amount = 0.35) {
      const ws = ctx.createWaveShaper();
      const k = 8 + amount * 22; // gentler saturation
      const n = 1024;
      const curve = new Float32Array(n);
      for (let i = 0; i < n; i++) {
        const x = (i * 2) / (n - 1) - 1;
        curve[i] = Math.tanh(k * x) / Math.tanh(k);
      }
      ws.curve = curve;
      ws.oversample = "2x";
      return ws;
    }

    async _ensure() {
      if (!this.enabled) return;
      if (typeof window === "undefined") return;

      if (!this._ctx) {
        this._ctx = this._audioContext();
        if (!this._ctx) return;

        this._master = this._ctx.createGain();
        this._master.gain.value = this.volume;
        this._master.connect(this._ctx.destination);

        const noiseBuf = this._makeNoiseBuffer(this._ctx);
        this._noiseSrc = this._ctx.createBufferSource();
        this._noiseSrc.buffer = noiseBuf;
        this._noiseSrc.loop = true;

        this._baseGain = this._ctx.createGain();
        this._baseGain.gain.value = 0.0;

        this._hitGain = this._ctx.createGain();
        this._hitGain.gain.value = 0.0;

        this._hp = this._ctx.createBiquadFilter();
        this._hp.type = "highpass";
        this._hp.frequency.value = 180;
        this._hp.Q.value = 0.7;

        this._bp = this._ctx.createBiquadFilter();
        this._bp.type = "bandpass";
        this._bp.frequency.value = 1400;
        this._bp.Q.value = 1.2;

        this._panner = this._ctx.createStereoPanner();
        this._panner.pan.value = 0;

        this._sat = this._makeSaturator(this._ctx, 0.30 + this.roughness * 0.25);

        this._comp = this._ctx.createDynamicsCompressor();
        this._comp.threshold.value = -30;
        this._comp.knee.value = 14;
        this._comp.ratio.value = 3.0;
        this._comp.attack.value = 0.006;
        this._comp.release.value = 0.12;

        // Very subtle flutter in timbre (not volume)
        this._lfo = this._ctx.createOscillator();
        this._lfo.type = "sine";
        this._lfo.frequency.value = 4.5;

        this._lfoGain = this._ctx.createGain();
        this._lfoGain.gain.value = 90 * this.roughness;

        this._lfo.connect(this._lfoGain);
        this._lfoGain.connect(this._bp.frequency);
        this._lfo.start();

        this._noiseSrc.connect(this._baseGain);
        this._noiseSrc.connect(this._hitGain);

        const merger = this._ctx.createGain();
        merger.gain.value = 1;

        this._baseGain.connect(merger);
        this._hitGain.connect(merger);

        merger.connect(this._hp);
        this._hp.connect(this._bp);
        this._bp.connect(this._sat);
        this._sat.connect(this._comp);
        this._comp.connect(this._panner);
        this._panner.connect(this._master);

        this._noiseSrc.start();
      }
    }

    async unlock() {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx) return;
      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}
    }

    autoUnlock(target = document) {
      if (!this.enabled || typeof window === "undefined") return () => {};
      const handler = async () => { await this.unlock(); cleanup(); };
      const cleanup = () => {
        target.removeEventListener("pointerdown", handler);
        target.removeEventListener("keydown", handler);
        target.removeEventListener("touchstart", handler);
      };
      target.addEventListener("pointerdown", handler, { once: true });
      target.addEventListener("touchstart", handler, { once: true });
      target.addEventListener("keydown", handler, { once: true });
      return cleanup;
    }

    async suspend() {
      if (!this.enabled || !this._ctx) return;
      this.penUp(true);
      try { if (this._ctx.state === "running") await this._ctx.suspend(); } catch (_) {}
    }

    async resume() {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx) return;
      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}
    }

    async penDown(intensity = 1) {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx || !this._baseGain || !this._hitGain || !this._bp || !this._panner) return;

      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}
      const t = this._ctx.currentTime;

      // Restore master volume if it got muted
      if (this._master && this._master.gain.value < 0.00001) {
        this._master.gain.cancelScheduledValues(t);
        this._master.gain.setValueAtTime(this._master.gain.value, t);
        this._master.gain.linearRampToValueAtTime(this.volume, t + 0.04);
      }

      const i = clamp(intensity, 0.15, 1);
      this._charCount += 1;

      // Gentle stereo drift
      this._panner.pan.setValueAtTime((Math.random() * 2 - 1) * this.stereo, t);

      // Smooth timbre changes (avoid clicky "per-letter" feel)
      const dt = t - (this._lastT || t);
      const smooth = dt < 0.14; // if letters are close together, keep sound continuous

      const targetFreq = smooth
        ? clamp(this._lastFreq + rand(-120, 120), 950, 2400)
        : clamp(1400 + rand(-350, 650) + this.roughness * rand(120, 520), 900, 2800);

      const targetQ = smooth
        ? clamp(this._lastQ + rand(-0.12, 0.12), 0.9, 2.2)
        : clamp(1.05 + rand(-0.15, 0.65) + this.roughness * 0.65, 0.8, 2.8);

      this._bp.frequency.cancelScheduledValues(t);
      this._bp.frequency.setValueAtTime(this._bp.frequency.value, t);
      this._bp.frequency.linearRampToValueAtTime(targetFreq, t + 0.06);

      this._bp.Q.cancelScheduledValues(t);
      this._bp.Q.setValueAtTime(this._bp.Q.value, t);
      this._bp.Q.linearRampToValueAtTime(targetQ, t + 0.06);

      this._lastFreq = targetFreq;
      this._lastQ = targetQ;
      this._lastT = t;

      // Continuous base friction (key to not sounding like a typewriter)
      const base = (this.baseLevel * 0.14) * i * (0.60 + 0.70 * this.roughness);
      this._baseGain.gain.cancelScheduledValues(t);
      this._baseGain.gain.setValueAtTime(this._baseGain.gain.value, t);
      this._baseGain.gain.linearRampToValueAtTime(base, t + this.baseAttack);

      // Small accents only every N letters (not every letter)
      const doHit = (this.transientLevel > 0) && (this._charCount % this.transientEvery === 0);
      if (doHit) {
        const hitPeak = (this.transientLevel * 0.16) * i * (0.60 + 0.55 * this.roughness);
        const rel = rand(this.hitReleaseMin, this.hitReleaseMax);
        this._hitGain.gain.cancelScheduledValues(t);
        this._hitGain.gain.setValueAtTime(this._hitGain.gain.value, t);
        this._hitGain.gain.linearRampToValueAtTime(hitPeak, t + this.hitAttack);
        this._hitGain.gain.linearRampToValueAtTime(0.0, t + this.hitAttack + rel);
      }

      // Rare squeak
      if (Math.random() < this.squeakChance * (0.45 + 0.5 * this.roughness)) {
        this._bp.frequency.setValueAtTime(targetFreq + rand(250, 700), t + 0.01);
        this._bp.frequency.setValueAtTime(targetFreq, t + 0.10);
      }
    }

    penUp(hard = false) {
      if (!this.enabled || !this._ctx || !this._baseGain || !this._hitGain) return;
      const t = this._ctx.currentTime;

      this._baseGain.gain.cancelScheduledValues(t);
      this._baseGain.gain.setValueAtTime(this._baseGain.gain.value, t);
      this._baseGain.gain.linearRampToValueAtTime(0.0, t + this.baseRelease);

      this._hitGain.gain.cancelScheduledValues(t);
      this._hitGain.gain.setValueAtTime(0.0, t);

      if (hard && this._master) {
        this._master.gain.cancelScheduledValues(t);
        this._master.gain.setValueAtTime(0.0, t);
      }
    }

    async hardStop() {
      this.penUp(true);
      if (this.allowSuspend) await this.suspend();
    }
  }


  class Handwriter {
    constructor(el, options = {}) {
      if (!el) throw new Error("HandwriterStylo.create(el) needs a DOM element.");
      injectStylesOnce();

      this.el = el;
      this.opts = normalizeOptions(options);

      this.el.classList.add("hw-stylo");
      if (this.opts.inkStyle) this.el.classList.add("hw-ink");
      if (this.opts.revealMode === "opacity") this.el.classList.add("hw-opacity");
      if (this.opts.revealMode === "pen") this.el.classList.add("hw-pen");

      this.el.style.setProperty("--hw-line-height", String(this.opts.lineHeight));
      this.el.style.setProperty("--hw-ink", this.opts.inkColor);
      this.el.style.setProperty("--hw-char-pad-x", this.opts.charPadX);
      this.el.style.setProperty("--hw-char-pad-y", this.opts.charPadY);
      this.el.style.setProperty("--hw-clip-pad-y", this.opts.clipPadY);
      this.el.style.setProperty("--hw-clip-pad-x", this.opts.clipPadX);

      if (this.opts.align) this.el.style.textAlign = this.opts.align;
      if (this.opts.fontFamily) this.el.style.fontFamily = this.opts.fontFamily;
      if (this.opts.fontSize) this.el.style.fontSize = this.opts.fontSize;

      this._pauseState = { paused: false };
      this._autoPaused = false;
      this._abort = null;
      this._queue = Promise.resolve();

      this.audio = new PenAudio(this.opts.sound);
      if (this.opts.sound.autoUnlock && typeof document !== "undefined") {
        this.audio.autoUnlock(document);
      }

      // Stop sound (and optionally pause writing) when tab is hidden / page loses focus.
      this._visibilityHandler = () => {
        if (typeof document === "undefined") return;
        if (document.hidden) {
          this.audio.penUp(false);
          if (this.opts.pauseWhenHidden && !this._pauseState.paused) {
            this._autoPaused = true;
            this.pause();
          }
          if (this.opts.sound?.allowSuspend) this.audio.suspend?.().catch(() => {});
        } else {
          // Try to resume audio if the browser suspended it while hidden.
          this.audio.resume?.().catch(() => {});
          if (this.opts.pauseWhenHidden && this._autoPaused) {
            this._autoPaused = false;
            this.resume();
          }
        }
      };

      this._blurHandler = () => {
        this.audio.penUp(false);
        if (this.opts.pauseWhenHidden && !this._pauseState.paused) {
          this._autoPaused = true;
          this.pause();
        }
        if (this.opts.sound?.allowSuspend) this.audio.suspend?.().catch(() => {});
      };

      this._focusHandler = () => {
        // Resume writing if we auto-paused on blur, and try to resume audio.
        this.audio.resume?.().catch(() => {});
        if (this.opts.pauseWhenHidden && this._autoPaused) {
          this._autoPaused = false;
          this.resume();
        }
      };

      if (typeof document !== "undefined") {
        document.addEventListener("visibilitychange", this._visibilityHandler);
      }
      if (typeof window !== "undefined") {
        window.addEventListener("blur", this._blurHandler);
        window.addEventListener("pagehide", this._blurHandler);
        window.addEventListener("focus", this._focusHandler);
        window.addEventListener("pageshow", this._focusHandler);
      }

      this.cursor = document.createElement("span");
      this.cursor.className = "hw-cursor";
      this.cursor.textContent = this.opts.cursorChar;
      if (this.opts.showCursor) this.el.appendChild(this.cursor);

      this._globalCharIndex = 0;
    }

    async unlockAudio() { await this.audio.unlock(); }

    pause() { this._pauseState.paused = true; }
    resume() { this._pauseState.paused = false; }

    writeOnUserGesture(content, writeOptions = {}) {
      const start = async () => {
        await this.unlockAudio();
        if (Array.isArray(content) && content.length && typeof content[0] === "object" && content[0] && ("text" in content[0])) {
          await this.writeRich(content, writeOptions);
        } else {
          await this.write(String(content ?? ""), writeOptions);
        }
      };

      const once = () => {
        document.removeEventListener("pointerdown", once);
        document.removeEventListener("touchstart", once);
        document.removeEventListener("keydown", once);
        start();
      };

      document.addEventListener("pointerdown", once, { once: true });
      document.addEventListener("touchstart", once, { once: true });
      document.addEventListener("keydown", once, { once: true });
    }

    clear() {
      this.el.textContent = "";
      // reset layout lock until next write
      this.el.style.minHeight = "";
      if (this.opts.showCursor) this.el.appendChild(this.cursor);
      this._globalCharIndex = 0;
    }


    _lockLayoutForBlocks(blocks) {
      if (!this.opts.lockLayout) return;
      if (typeof document === "undefined") return;

      // Only makes sense when the container is already on the page and has a width.
      const width = this.el.clientWidth;
      if (!width || width < 20) return;

      // Build a hidden measurement node with the same typography + block styles.
      const meas = document.createElement("div");
      meas.style.position = "absolute";
      meas.style.visibility = "hidden";
      meas.style.pointerEvents = "none";
      meas.style.left = "-99999px";
      meas.style.top = "0";
      meas.style.width = width + "px";
      meas.style.whiteSpace = "pre-wrap";

      // Copy key typography from the writer element.
      const cs = window.getComputedStyle(this.el);
      meas.style.fontFamily = cs.fontFamily;
      meas.style.fontSize = cs.fontSize;
      meas.style.lineHeight = cs.lineHeight;
      meas.style.letterSpacing = cs.letterSpacing;
      meas.style.fontFeatureSettings = cs.fontFeatureSettings;
      meas.style.fontKerning = cs.fontKerning;

      for (const b of blocks) {
        const line = document.createElement("div");
        line.style.display = "block";
        line.style.width = "100%";

        const size = (b.size ?? 1);
        const lh = (b.lineHeight ?? this.opts.lineHeight);
        const mt = (b.marginTop ?? 0);
        const mb = (b.marginBottom ?? 0);
        const align = (b.align ?? this.opts.align);

        // Replicate the library block sizing logic
        line.style.fontSize = `calc(${this.opts.fontSize} * ${size})`;
        line.style.lineHeight = String(lh);
        line.style.marginTop = mt + "px";
        line.style.marginBottom = mb + "px";
        if (align) line.style.textAlign = align;

        line.textContent = String(b.text ?? "");
        meas.appendChild(line);
      }

      document.body.appendChild(meas);
      const h = Math.ceil(meas.getBoundingClientRect().height);
      meas.remove();

      if (h > 0) {
        // Fix the element height from the start so centered layouts don't move as content grows.
        this.el.style.minHeight = h + "px";
      }
    }


    write(text, options = {}) {
      return this.writeRich([{ text: String(text ?? "") }], options);
    }

    writeRich(blocks, options = {}) {
      const mode = options.mode || this.opts.mode;
      const reduced =
        (this.opts.reducedMotion === true) ||
        (this.opts.reducedMotion !== false && prefersReducedMotion());

      this._queue = this._queue.then(async () => {
        if (this._abort) this._abort.abort();
        this._abort = new AbortController();
        const signal = this._abort.signal;

        if (mode === "replace") this.clear();

        // Prevent vertical-centering layouts from shifting while text grows
        if (mode === "replace") this._lockLayoutForBlocks(blocks);

        if (reduced) {
          for (const b of blocks) {
            const lineEl = this._createBlockEl(b);
            lineEl.textContent = String(b.text ?? "");
            this.el.insertBefore(lineEl, this.opts.showCursor ? this.cursor : null);
          }
          return;
        }

        let randomPausesUsed = 0;

        for (const block of blocks) {
          if (signal.aborted) break;

          const blockEl = this._createBlockEl(block);
          this.el.insertBefore(blockEl, this.opts.showCursor ? this.cursor : null);
          await nextPaintFrame();

          const tokens = [...String(block.text ?? "")];

          // Per-block timing overrides (so each line can have its own speed / end pause)
          const blockTimeScale = Math.max(0.05, (block.timeScale ?? this.opts.timeScale));
          const blockMinDelay = Math.max(0, (block.minDelay ?? this.opts.minDelay));
          const blockMaxDelay = Math.max(blockMinDelay, (block.maxDelay ?? this.opts.maxDelay));
          const blockLinePauseMin = Math.max(0, (block.linePauseMin ?? this.opts.linePauseMin));
          const blockLinePauseMax = Math.max(blockLinePauseMin, (block.linePauseMax ?? this.opts.linePauseMax));


          // Track word starts to reduce left-jitter/clipping on swashy fonts
          let wordStart = true;

          // Word container to prevent mid-word breaks
          let wordEl = null;
          const ensureWord = () => {
            if (!wordEl) {
              wordEl = document.createElement("span");
              wordEl.className = "hw-word";
              blockEl.appendChild(wordEl);
            }
            return wordEl;
          };
          const endWord = () => { wordEl = null; };

          for (let i = 0; i < tokens.length; i++) {
            const ch = tokens[i];
            if (signal.aborted) break;

            // If lockLayoutX is enabled, we DON'T reveal while building; we pre-create all spans first
            // so centered lines don't shift as they grow.
            if (this.opts.lockLayoutX || block.lockLayoutX) {
              // We'll build everything once, then reveal in a second pass.
              break;
            }

            if (ch === "\n") {
              endWord();
              wordStart = true;
              blockEl.appendChild(document.createElement("br"));
              this._globalCharIndex += 1;
              await nextPaintFrame();
              continue;
            }

            // Spaces: allow wrapping (normal space)
            if (ch === " " || ch === "\t") {
              endWord();
              wordStart = true;
              blockEl.appendChild(document.createTextNode(ch));
              this.audio.penUp();

              let delay = rand(blockMinDelay, blockMaxDelay) * blockTimeScale;
              if (this.opts.randomPauses && randomPausesUsed < this.opts.maxRandomPauses && Math.random() < this.opts.pauseChanceOnSpace) {
                randomPausesUsed += 1;
                delay += rand(this.opts.pauseMin, this.opts.pauseMax) * blockTimeScale;
              }

              this._globalCharIndex += 1;
              await wait(delay, signal, this._pauseState);
              continue;
            }

            const parent = ensureWord();
            const span = this._makeCharSpan(ch, this._globalCharIndex, wordStart, blockTimeScale);
            parent.appendChild(span);
            wordStart = false;

            // Reveal now
            span.classList.add("hw-on");
            span.getBoundingClientRect();
            await nextPaintFrame();

            const intensity = /[A-ZÁÉÍÓÚÜÑ0-9]/i.test(ch) ? 0.92 : 0.75;
            await this.audio.penDown(intensity);

            let delay = rand(blockMinDelay, blockMaxDelay) * blockTimeScale;

            if (this.opts.punctuationPauses && this.opts.punctuationSet.has(ch)) {
              const extra = rand(this.opts.punctuationMin, this.opts.punctuationMax) * blockTimeScale;
              delay += extra;
              if (extra > 260) this.audio.penUp();
            }

            // Avoid long pauses inside a word; allow only after punctuation and within budget
            if (this.opts.randomPauses && randomPausesUsed < this.opts.maxRandomPauses && this.opts.punctuationSet.has(ch) && Math.random() < this.opts.pauseChance) {
              randomPausesUsed += 1;
              this.audio.penUp();
              delay += rand(this.opts.pauseMin, this.opts.pauseMax) * blockTimeScale;
            }

            this._globalCharIndex += 1;
            await wait(delay, signal, this._pauseState);
          }

          // lockLayoutX branch: pre-create all spans, then reveal sequentially
          if (this.opts.lockLayoutX || block.lockLayoutX) {
            const steps = [];
            // reset any partially built content in this block
            blockEl.textContent = "";
            wordEl = null;
            wordStart = true;

            const ensureWord2 = () => {
              if (!wordEl) {
                wordEl = document.createElement("span");
                wordEl.className = "hw-word";
                blockEl.appendChild(wordEl);
              }
              return wordEl;
            };
            const endWord2 = () => { wordEl = null; };

            for (let i = 0; i < tokens.length; i++) {
              const ch = tokens[i];

              if (ch === "\n") {
                endWord2();
                wordStart = true;
                blockEl.appendChild(document.createElement("br"));
                steps.push({ type: "nl" });
                this._globalCharIndex += 1;
                continue;
              }

              if (ch === " " || ch === "\t") {
                endWord2();
                wordStart = true;
                blockEl.appendChild(document.createTextNode(ch));
                steps.push({ type: "space", ch });
                this._globalCharIndex += 1;
                continue;
              }

              const parent = ensureWord2();
              const span = this._makeCharSpan(ch, this._globalCharIndex, wordStart, blockTimeScale);
              parent.appendChild(span); // stays invisible until hw-on
              steps.push({ type: "char", ch, span });
              wordStart = false;
              this._globalCharIndex += 1;
            }

            await nextPaintFrame();

            for (let s = 0; s < steps.length; s++) {
              if (signal.aborted) break;
              const step = steps[s];

              if (step.type === "nl") {
                this.audio.penUp();
                await nextPaintFrame();
                continue;
              }

              if (step.type === "space") {
                this.audio.penUp();
                let delay = rand(blockMinDelay, blockMaxDelay) * blockTimeScale;
                if (this.opts.randomPauses && randomPausesUsed < this.opts.maxRandomPauses && Math.random() < this.opts.pauseChanceOnSpace) {
                  randomPausesUsed += 1;
                  delay += rand(this.opts.pauseMin, this.opts.pauseMax) * blockTimeScale;
                }
                await wait(delay, signal, this._pauseState);
                continue;
              }

              // reveal char
              step.span.classList.add("hw-on");
              step.span.getBoundingClientRect();
              await nextPaintFrame();

              const intensity = /[A-ZÁÉÍÓÚÜÑ0-9]/i.test(step.ch) ? 0.92 : 0.75;
              await this.audio.penDown(intensity);

              let delay = rand(blockMinDelay, blockMaxDelay) * blockTimeScale;

              if (this.opts.punctuationPauses && this.opts.punctuationSet.has(step.ch)) {
                const extra = rand(this.opts.punctuationMin, this.opts.punctuationMax) * blockTimeScale;
                delay += extra;
                if (extra > 260) this.audio.penUp();
              }

              if (this.opts.randomPauses && randomPausesUsed < this.opts.maxRandomPauses && this.opts.punctuationSet.has(step.ch) && Math.random() < this.opts.pauseChance) {
                randomPausesUsed += 1;
                this.audio.penUp();
                delay += rand(this.opts.pauseMin, this.opts.pauseMax) * blockTimeScale;
              }

              await wait(delay, signal, this._pauseState);
            }
          }

          // Optional pause at end of this block/line
          const lpMin = Math.max(0, blockLinePauseMin);
          const lpMax = Math.max(lpMin, blockLinePauseMax);
          if (lpMax > 0) {
            await wait(rand(lpMin, lpMax) * blockTimeScale, signal, this._pauseState);
          }

          this.audio.penUp();
        }
        // Ensure sound is fully stopped after finishing writing (no lingering tab audio icon).
        if (this.opts.sound?.stopOnFinish) {
          this.audio.hardStop?.().catch(() => {});
        } else {
          this.audio.penUp(false);
        }

      });

      return this._queue;
    }

    _createBlockEl(block) {
      const el = document.createElement("div");
      el.className = "hw-line";

      const size = block.size ?? 1;
      const lh = block.lineHeight ?? this.opts.lineHeight;
      const mt = block.marginTop ?? 0;
      const mb = block.marginBottom ?? 0;
      const align = block.align ?? this.opts.align;

      el.style.fontSize = `calc(${this.opts.fontSize} * ${size})`;
      el.style.lineHeight = String(lh);
      el.style.marginTop = `${mt}px`;
      el.style.marginBottom = `${mb}px`;
      if (align) el.style.textAlign = align;

      return el;
    }

    _makeCharSpan(ch, globalIndex, isWordStart = false, timeScale = this.opts.timeScale) {
      const span = document.createElement("span");
      span.className = "hw-char";
      span.textContent = ch;

      const reveal = clamp(rand(this.opts.revealMin, this.opts.revealMax) * timeScale, 45, 1200);
      span.style.setProperty("--hw-reveal-ms", `${reveal}ms`);
      span.style.setProperty("--hw-fade-ms", `${Math.min(reveal, 150)}ms`);

      const pressure = 1 - rand(0, this.opts.pressureJitter);
      span.style.setProperty("--hw-op", String(clamp(pressure, 0.72, 1)));

      // Pen-like reveal parameters (used when revealMode === "pen")
      if (this.opts.revealMode === "pen") {
        span.style.setProperty("--hw-pen-angle", `${rand(100, 118)}deg`);
        span.style.setProperty("--hw-pen-edge", `${rand(42, 52)}%`);
        span.style.setProperty("--hw-pen-y", `${rand(-8, 8)}%`);
        span.style.setProperty("--hw-pen-y2", `${rand(-6, 6)}%`);

        // Choose a slight variation of pen direction (up/down diagonal)
        const isDesc = /[gjpqy]/i.test(ch);
        const pUp = isDesc ? 0.25 : 0.55;
        if (Math.random() < pUp) span.classList.add("hw-pen-up");

        // Vertical variation: some letters start top-first (ascenders/caps), others bottom-first.
        const isAsc = /[bdfhklt]/i.test(ch) || /[A-ZÁÉÍÓÚÜÑ]/.test(ch);
        const pTop = isAsc ? 0.65 : (isDesc ? 0.20 : 0.35);
        if (Math.random() < pTop) span.classList.add("hw-pen-top");
      }

      // Reduce jitter on the first letter of each word (prevents left-edge clipping on swashy fonts)
      const j = this.opts.jitter;
      const startFactor = (isWordStart || globalIndex < 3) ? 0.18 : 1.0;
      const jx = j.x * startFactor;
      const jy = j.y * startFactor;
      const jr = j.rotate * startFactor;

      const curve = this.opts.curve.enabled
        ? (this.opts.curve.amplitude * Math.sin((globalIndex / this.opts.curve.period) * Math.PI * 2))
        : 0;

      const tx = rand(-jx, jx);
      const ty = rand(-jy, jy) + curve;
      const rot = rand(-jr, jr);

      span.style.transform = `translate(${tx}px, ${ty}px) rotate(${rot}deg)`;
      return span;
    }
  }

  function normalizeOptions(o) {
    const opts = {
      mode: o.mode || "replace",

      fontFamily: o.fontFamily || `'Parisienne', 'Italianno', 'Allura', 'Segoe Script', cursive`,
      fontSize: o.fontSize || "44px",
      align: o.align || "center",
      lineHeight: o.lineHeight ?? 1.20,

      // Multiply all timing (writing + pauses + reveal) by this factor. >1 = slower.
      timeScale: o.timeScale ?? 1,

      inkColor: o.inkColor || "rgba(12,12,12,0.82)",
      inkStyle: o.inkStyle ?? true,

      // "clip" (default) = simple ink reveal (fast)
      // "pen" = diagonal moving front (more like a pen stroke)
      // "opacity" = safest (no clipping), but less "ink reveal"
      revealMode: o.revealMode ?? "clip",

      // NEW: separate X/Y pads + clip pad Y
      charPadX: o.charPadX ?? "0.32em",
      charPadY: o.charPadY ?? "0.16em",
      clipPadY: o.clipPadY ?? "-0.38em",
      // Extra left expansion for swashy fonts (prevents left-cut on letters like g/y)
      clipPadX: o.clipPadX ?? "-0.45em",

      pressureJitter: clamp(o.pressureJitter ?? 0.12, 0, 0.5),

      jitter: {
        x: o.jitter?.x ?? 0.22,
        y: o.jitter?.y ?? 0.34,
        rotate: o.jitter?.rotate ?? 0.5,
      },
      curve: {
        enabled: o.curve?.enabled ?? true,
        amplitude: o.curve?.amplitude ?? 0.55,
        period: o.curve?.period ?? 34,
      },

      minDelay: o.minDelay ?? 34,
      maxDelay: o.maxDelay ?? 95,
      revealMin: o.revealMin ?? 85,
      revealMax: o.revealMax ?? 165,

      randomPauses: o.randomPauses ?? true,
      // Maximum number of long "thinking" pauses per whole message.
      maxRandomPauses: o.maxRandomPauses ?? 2,

      // Extra pause after each block/line (ms). Can be overridden per block.
      linePauseMin: o.linePauseMin ?? 0,
      linePauseMax: o.linePauseMax ?? 0,
      pauseChance: clamp(o.pauseChance ?? 0.18, 0, 1),
      pauseChanceOnSpace: clamp(o.pauseChanceOnSpace ?? 0.10, 0, 1),
      pauseMin: o.pauseMin ?? 240,
      pauseMax: o.pauseMax ?? 1400,

      punctuationPauses: o.punctuationPauses ?? true,
      punctuationMin: o.punctuationMin ?? 160,
      punctuationMax: o.punctuationMax ?? 520,
      punctuationSet: new Set((o.punctuationChars || ".,;:!?").split("")),

      showCursor: o.showCursor ?? true,
      cursorChar: o.cursorChar || "▏",

      sound: {
        enabled: o.sound?.enabled ?? true,
        volume: clamp(o.sound?.volume ?? 0.16, 0, 1),
        autoUnlock: o.sound?.autoUnlock ?? true,

        // Sound character (0..1)
        roughness: o.sound?.roughness ?? 0.65,
        squeakChance: o.sound?.squeakChance ?? 0.06,
        stereo: o.sound?.stereo ?? 0.18,

        // Set baseLevel=0 to remove continuous noise while writing.
        baseLevel: o.sound?.baseLevel ?? 0.06,
        transientLevel: o.sound?.transientLevel ?? 0.14,

        // End-of-writing stop (default true)
        stopOnFinish: o.sound?.stopOnFinish ?? true,

        // Suspending can break on Safari; default false. If true, removes any chance of a tab audio indicator.
        allowSuspend: o.sound?.allowSuspend ?? false,
      },

      reducedMotion: (o.reducedMotion ?? null),

      // If true, pre-measure final block height and set minHeight so centered layouts don't shift while writing.
      lockLayout: o.lockLayout ?? true,
      // If true, pre-creates invisible characters so centered text doesn't shift horizontally while writing.
      lockLayoutX: o.lockLayoutX ?? true,

      // If true, writing pauses when the tab is hidden (and sound is muted/suspended).
      pauseWhenHidden: o.pauseWhenHidden ?? true,
    };

    opts.minDelay = clamp(opts.minDelay, 0, 5000);
    opts.maxDelay = clamp(opts.maxDelay, opts.minDelay, 8000);
    opts.pauseMin = clamp(opts.pauseMin, 0, 20000);
    opts.pauseMax = clamp(opts.pauseMax, opts.pauseMin, 60000);
    return opts;
  }

  function create(el, options) { return new Handwriter(el, options); }
  return { create };
});
