/*!
 * HandwriterStylo Canvas v0.1.0
 * Canvas "ink brush" reveal that feels like a pen writing live.
 * - Works with any loaded web font (including BrittanySignature via @font-face).
 * - Reveals text through an accumulating brush mask (not a left-to-right wipe).
 * - Supports per-block timing: minDelay/maxDelay, wordPauseMin/Max, punctuationMin/Max, linePauseMin/Max, revealMin/Max
 *
 * MIT License
 */
(function (root, factory) {
  if (typeof module === "object" && module.exports) module.exports = factory();
  else root.HandwriterStyloCanvas = factory();
})(typeof self !== "undefined" ? self : this, function () {
  "use strict";

  const clamp = (n, a, b) => Math.max(a, Math.min(b, n));
  const rand = (a, b) => a + Math.random() * (b - a);

  function prefersReducedMotion() {
    return typeof window !== "undefined"
      && window.matchMedia
      && window.matchMedia("(prefers-reduced-motion: reduce)").matches;
  }

  // ---- Optional pen audio (very lightweight; same idea as your previous versions) ----
  class PenAudio {
    constructor(opts = {}) {
      this.enabled = !!opts.enabled;
      this.volume = clamp(opts.volume ?? 0.16, 0, 1);
      this.roughness = clamp(opts.roughness ?? 0.50, 0, 1);
      this.baseLevel = clamp(opts.baseLevel ?? 0.08, 0, 1);
      this.transientLevel = clamp(opts.transientLevel ?? 0.10, 0, 1);
      this.transientEvery = Math.max(1, opts.transientEvery ?? 4);

      this._ctx = null;
      this._master = null;
      this._noiseSrc = null;
      this._baseGain = null;
      this._hitGain = null;
      this._bp = null;
      this._hp = null;
      this._panner = null;
      this._charCount = 0;
    }

    _audioContext() {
      const AudioCtx = window.AudioContext || window.webkitAudioContext;
      return AudioCtx ? new AudioCtx() : null;
    }

    _makeNoiseBuffer(ctx) {
      const seconds = 1.0;
      const buffer = ctx.createBuffer(1, Math.floor(ctx.sampleRate * seconds), ctx.sampleRate);
      const data = buffer.getChannelData(0);
      for (let i = 0; i < data.length; i++) data[i] = (Math.random() * 2 - 1);
      return buffer;
    }

    async _ensure() {
      if (!this.enabled) return;
      if (typeof window === "undefined") return;

      if (!this._ctx) {
        this._ctx = this._audioContext();
        if (!this._ctx) return;

        this._master = this._ctx.createGain();
        this._master.gain.value = this.volume;
        this._master.connect(this._ctx.destination);

        const noiseBuf = this._makeNoiseBuffer(this._ctx);
        this._noiseSrc = this._ctx.createBufferSource();
        this._noiseSrc.buffer = noiseBuf;
        this._noiseSrc.loop = true;

        this._baseGain = this._ctx.createGain();
        this._baseGain.gain.value = 0.0;

        this._hitGain = this._ctx.createGain();
        this._hitGain.gain.value = 0.0;

        this._hp = this._ctx.createBiquadFilter();
        this._hp.type = "highpass";
        this._hp.frequency.value = 180;
        this._hp.Q.value = 0.7;

        this._bp = this._ctx.createBiquadFilter();
        this._bp.type = "bandpass";
        this._bp.frequency.value = 1400;
        this._bp.Q.value = 1.2;

        this._panner = this._ctx.createStereoPanner();
        this._panner.pan.value = 0;

        this._noiseSrc.connect(this._baseGain);
        this._noiseSrc.connect(this._hitGain);

        const merger = this._ctx.createGain();
        this._baseGain.connect(merger);
        this._hitGain.connect(merger);

        merger.connect(this._hp);
        this._hp.connect(this._bp);
        this._bp.connect(this._panner);
        this._panner.connect(this._master);

        this._noiseSrc.start();
      }
    }

    async unlock() {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx) return;
      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}
    }

    autoUnlock(target = document) {
      if (!this.enabled || typeof window === "undefined") return () => {};
      const handler = async () => { await this.unlock(); cleanup(); };
      const cleanup = () => {
        target.removeEventListener("pointerdown", handler);
        target.removeEventListener("keydown", handler);
        target.removeEventListener("touchstart", handler);
      };
      target.addEventListener("pointerdown", handler, { once: true });
      target.addEventListener("touchstart", handler, { once: true });
      target.addEventListener("keydown", handler, { once: true });
      return cleanup;
    }

    async penDown(intensity = 1) {
      if (!this.enabled) return;
      await this._ensure();
      if (!this._ctx || !this._baseGain || !this._hitGain || !this._bp || !this._panner) return;

      try { if (this._ctx.state !== "running") await this._ctx.resume(); } catch (_) {}
      const t = this._ctx.currentTime;

      const i = clamp(intensity, 0.15, 1);
      this._charCount += 1;

      // gentle stereo drift
      this._panner.pan.setValueAtTime((Math.random() * 2 - 1) * 0.12, t);

      // smooth timbre
      const baseFreq = clamp(1250 + rand(-220, 380) + this.roughness * rand(80, 420), 900, 2600);
      this._bp.frequency.setValueAtTime(baseFreq, t);
      this._bp.Q.setValueAtTime(clamp(1.0 + rand(-0.1, 0.6) + this.roughness * 0.5, 0.8, 2.2), t);

      // continuous friction
      const base = (this.baseLevel * 0.14) * i * (0.62 + 0.6 * this.roughness);
      this._baseGain.gain.cancelScheduledValues(t);
      this._baseGain.gain.setValueAtTime(this._baseGain.gain.value, t);
      this._baseGain.gain.linearRampToValueAtTime(base, t + 0.04);

      // small accent every N letters
      if (this.transientLevel > 0 && (this._charCount % this.transientEvery === 0)) {
        const hitPeak = (this.transientLevel * 0.14) * i;
        this._hitGain.gain.cancelScheduledValues(t);
        this._hitGain.gain.setValueAtTime(this._hitGain.gain.value, t);
        this._hitGain.gain.linearRampToValueAtTime(hitPeak, t + 0.02);
        this._hitGain.gain.linearRampToValueAtTime(0.0, t + 0.22);
      }
    }

    penUp() {
      if (!this.enabled || !this._ctx || !this._baseGain || !this._hitGain) return;
      const t = this._ctx.currentTime;
      this._baseGain.gain.cancelScheduledValues(t);
      this._baseGain.gain.setValueAtTime(this._baseGain.gain.value, t);
      this._baseGain.gain.linearRampToValueAtTime(0.0, t + 0.10);

      this._hitGain.gain.cancelScheduledValues(t);
      this._hitGain.gain.setValueAtTime(0.0, t);
    }

    stop() {
      this.penUp();
    }
  }

  // ---- Canvas writer ----
  class CanvasWriter {
    constructor(container, opts = {}) {
      if (!container) throw new Error("CanvasWriter needs a container element");
      this.el = container;
      this.opts = normalizeOptions(opts);

      this._abort = null;
      this._paused = false;
      this._autoPaused = false;

      // Canvas elements
      this.canvas = document.createElement("canvas");
      this.canvas.style.width = "100%";
      this.canvas.style.height = "100%";
      this.canvas.style.display = "block";
      this.canvas.style.pointerEvents = "none";
      this.el.innerHTML = "";
      this.el.appendChild(this.canvas);

      this.ctx = this.canvas.getContext("2d");

      // Offscreen buffers
      this.textCanvas = document.createElement("canvas");
      this.textCtx = this.textCanvas.getContext("2d");

      this.maskCanvas = document.createElement("canvas");
      this.maskCtx = this.maskCanvas.getContext("2d");

      this.audio = new PenAudio(this.opts.sound);
      if (this.opts.sound.autoUnlock) this.audio.autoUnlock(document);

      // Resize handling
      this._ro = new ResizeObserver(() => this._resize());
      this._ro.observe(this.el);
      this._resize();

      // visibility/focus
      this._visibilityHandler = () => {
        if (document.hidden) {
          if (this.opts.pauseWhenHidden && !this._paused) { this._autoPaused = true; this.pause(); }
          this.audio.penUp();
        } else {
          if (this.opts.pauseWhenHidden && this._autoPaused) { this._autoPaused = false; this.resume(); }
        }
      };
      document.addEventListener("visibilitychange", this._visibilityHandler);

      this._blurHandler = () => {
        if (this.opts.pauseWhenHidden && !this._paused) { this._autoPaused = true; this.pause(); }
        this.audio.penUp();
      };
      this._focusHandler = () => {
        if (this.opts.pauseWhenHidden && this._autoPaused) { this._autoPaused = false; this.resume(); }
      };
      window.addEventListener("blur", this._blurHandler);
      window.addEventListener("focus", this._focusHandler);
    }

    destroy() {
      try { this._ro?.disconnect(); } catch {}
      document.removeEventListener("visibilitychange", this._visibilityHandler);
      window.removeEventListener("blur", this._blurHandler);
      window.removeEventListener("focus", this._focusHandler);
      this.stop();
      this.el.innerHTML = "";
    }

    _resize() {
      const dpr = Math.max(1, window.devicePixelRatio || 1);
      const rect = this.el.getBoundingClientRect();
      const w = Math.max(2, Math.floor(rect.width * dpr));
      const h = Math.max(2, Math.floor(rect.height * dpr));

      this.canvas.width = w;
      this.canvas.height = h;
      this.textCanvas.width = w;
      this.textCanvas.height = h;
      this.maskCanvas.width = w;
      this.maskCanvas.height = h;

      // set scaled coordinate system
      this.ctx.setTransform(1,0,0,1,0,0);
      this.textCtx.setTransform(1,0,0,1,0,0);
      this.maskCtx.setTransform(1,0,0,1,0,0);

      this.ctx.scale(dpr, dpr);
      this.textCtx.scale(dpr, dpr);
      this.maskCtx.scale(dpr, dpr);

      this._dpr = dpr;
      this._w = rect.width;
      this._h = rect.height;
    }

    pause() { this._paused = true; }
    resume() { this._paused = false; }

    stop() {
      if (this._abort) this._abort.abort();
      this._abort = null;
      this.audio.stop();
    }

    clear() {
      this.stop();
      this.maskCtx.clearRect(0, 0, this._w, this._h);
      this.ctx.clearRect(0, 0, this._w, this._h);
      this.textCtx.clearRect(0, 0, this._w, this._h);
    }

    async unlockAudio() { await this.audio.unlock(); }


    async ensureFontsLoaded(blocks) {
      if (typeof document === "undefined" || !document.fonts) return;
      const samples = [];

      // Gather font declarations we will use (base + any block overrides)
      for (const b of (blocks || [])) {
        const family = (b.fontFamily ?? this.opts.fontFamily);
        const size = this._resolveBlockFontSize(b);
        const font = `${this.opts.fontStyle} ${this.opts.fontWeight} ${size} ${family}`;
        samples.push(font);
      }
      // Also include base font as fallback
      samples.push(`${this.opts.fontStyle} ${this.opts.fontWeight} ${this.opts.fontSize} ${this.opts.fontFamily}`);

      // Ask browser to load them
      try {
        await Promise.all(samples.map(f => document.fonts.load(f, "Hola Madrid")));
        await document.fonts.ready;
      } catch (_) {
        // ignore – will fall back to default fonts
      }
    }


    writeOnUserGesture(blocks, options = {}) {
      const once = async () => {
        document.removeEventListener("pointerdown", once);
        document.removeEventListener("touchstart", once);
        document.removeEventListener("keydown", once);
        await this.unlockAudio();
        if (this.opts.autoWaitForFonts) await this.ensureFontsLoaded(blocks);
        await this.writeRich(blocks, options);
      };
      document.addEventListener("pointerdown", once, { once: true });
      document.addEventListener("touchstart", once, { once: true });
      document.addEventListener("keydown", once, { once: true });
    }

    async writeRich(blocks, options = {}) {
      const mode = options.mode || this.opts.mode;
      if (mode === "replace") this.clear();

      const reduced = (this.opts.reducedMotion === true) || (this.opts.reducedMotion !== false && prefersReducedMotion());
      if (reduced) {
        // Just render final text
        this._renderFinalText(blocks);
        this.ctx.clearRect(0,0,this._w,this._h);
        this.ctx.drawImage(this.textCanvas, 0, 0, this._w, this._h);
        return;
      }

      this._abort = new AbortController();
      const signal = this._abort.signal;

      // Ensure fonts are loaded before rendering to canvas (otherwise it draws with fallback)
      if (this.opts.autoWaitForFonts) await this.ensureFontsLoaded(blocks);

      // Pre-render full letter to textCanvas
      const layout = this._renderFinalText(blocks);

      // Mask starts empty
      this.maskCtx.clearRect(0, 0, this._w, this._h);

      // Draw loop (composite text with mask)
      const composite = () => {
        this.ctx.clearRect(0, 0, this._w, this._h);

        // Draw full text first
        this.ctx.drawImage(this.textCanvas, 0, 0, this._w, this._h);

        // Keep only masked region
        this.ctx.globalCompositeOperation = "destination-in";
        this.ctx.drawImage(this.maskCanvas, 0, 0, this._w, this._h);
        this.ctx.globalCompositeOperation = "source-over";
      };

      // Build events (letters + pauses)
      const events = this._buildEvents(blocks, layout);

      // Execute events
      for (const ev of events) {
        if (signal.aborted) break;

        // Pause handling
        while (this._paused && !signal.aborted) {
          await new Promise(r => setTimeout(r, 50));
        }
        if (signal.aborted) break;

        if (ev.type === "pause") {
          this.audio.penUp();
          await sleep(ev.ms, signal);
          continue;
        }

        if (ev.type === "letter") {
          // Ink this letter with a brush polyline over duration
          await this._inkLetter(ev, composite, signal);
        }
      }

      this.audio.penUp();
    }

    _renderFinalText(blocks) {
      const ctx = this.textCtx;
      ctx.clearRect(0, 0, this._w, this._h);

      const ink = this.opts.inkColor;
      ctx.fillStyle = ink;

      // layout blocks vertically centered by default
      const lines = [];
      let totalH = 0;

      // First pass: measure each line
      for (const b of blocks) {
        const fontSize = parseCssPx(this._resolveBlockFontSize(b));
        const lh = (b.lineHeight ?? this.opts.lineHeight);
        const mt = (b.marginTop ?? 0);
        const mb = (b.marginBottom ?? 0);
        const lineH = fontSize * lh;

        const font = this._resolveFont(b);
        ctx.font = font;

        const text = String(b.text ?? "");
        const width = ctx.measureText(text).width;

        lines.push({
          text,
          font,
          fontSize,
          lineHeight: lineH,
          mt, mb,
          width,
          align: (b.align ?? this.opts.align),
        });

        totalH += mt + lineH + mb;
      }

      const startY = this.opts.verticalAlign === "top"
        ? 0
        : (this._h - totalH) / 2;

      // Draw pass
      let y = startY;
      const layout = [];
      for (let i = 0; i < lines.length; i++) {
        const L = lines[i];
        y += L.mt;

        ctx.font = L.font;
        ctx.fillStyle = ink;

        const baselineY = y + L.fontSize; // approx baseline
        const x = (L.align === "center")
          ? (this._w / 2 - L.width / 2)
          : (L.align === "right")
            ? (this._w - L.width)
            : 0;

        // actual fillText baseline is alphabetic; baselineY works well for cursive
        ctx.textBaseline = "alphabetic";
        ctx.fillText(L.text, x, baselineY);

        layout.push({ x, baselineY, font: L.font, fontSize: L.fontSize, text: L.text, align: L.align });

        y += L.lineHeight + L.mb;
      }

      return layout;
    }

    _buildEvents(blocks, layout) {
      const events = [];
      const punctuation = this.opts.punctuationSet;

      for (let bi = 0; bi < blocks.length; bi++) {
        const b = blocks[bi];
        const line = layout[bi];

        const minDelay = b.minDelay ?? this.opts.minDelay;
        const maxDelay = b.maxDelay ?? this.opts.maxDelay;
        const timeScale = b.timeScale ?? this.opts.timeScale;

        const revealMin = b.revealMin ?? this.opts.revealMin;
        const revealMax = b.revealMax ?? this.opts.revealMax;

        const punctMin = b.punctuationMin ?? this.opts.punctuationMin;
        const punctMax = b.punctuationMax ?? this.opts.punctuationMax;

        const wordPauseMin = b.wordPauseMin ?? this.opts.wordPauseMin;
        const wordPauseMax = b.wordPauseMax ?? this.opts.wordPauseMax;

        const linePauseMin = b.linePauseMin ?? this.opts.linePauseMin;
        const linePauseMax = b.linePauseMax ?? this.opts.linePauseMax;

        // measure per-character x positions using measureText of incremental substring (simple + reliable)
        const ctx = this.textCtx;
        ctx.font = line.font;
        const full = line.text;
        let prevWidth = 0;

        for (let i = 0; i < full.length; i++) {
          const ch = full[i];

          const sub = full.slice(0, i + 1);
          const w = ctx.measureText(sub).width;
          const chW = w - prevWidth;

          const x0 = line.x + prevWidth;
          prevWidth = w;

          if (ch === " " || ch === "\t") {
            // word boundary pause (extra)
            const extra = rand(wordPauseMin, wordPauseMax) * timeScale;
            if (extra > 0) events.push({ type: "pause", ms: extra });
            continue;
          }

          // letter ink event
          const dur = rand(revealMin, revealMax) * timeScale;
          const gap = rand(minDelay, maxDelay) * timeScale;

          events.push({
            type: "letter",
            ch,
            x: x0,
            w: chW,
            baselineY: line.baselineY,
            font: line.font,
            fontSize: line.fontSize,
            duration: dur,
          });

          // punctuation pause after punctuation
          if (this.opts.punctuationPauses && punctuation.has(ch)) {
            const p = rand(punctMin, punctMax) * timeScale;
            if (p > 0) events.push({ type: "pause", ms: p });
          } else {
            // small gap between letters
            if (gap > 0) events.push({ type: "pause", ms: gap });
          }
        }

        // end-of-line pause
        const lp = rand(linePauseMin, linePauseMax) * timeScale;
        if (lp > 0) events.push({ type: "pause", ms: lp });
      }

      return events;
    }

    async _inkLetter(ev, composite, signal) {
      const ctx = this.maskCtx;
      const textCtx = this.textCtx;

      // brush style
      const penW = clamp(this.opts.penWidth, 1.2, 12);
      ctx.lineCap = "round";
      ctx.lineJoin = "round";
      ctx.strokeStyle = "rgba(255,255,255,1)";
      ctx.fillStyle = "rgba(255,255,255,1)";

      // Get metrics for vertical behavior (asc/desc)
      textCtx.font = ev.font;
      const m = textCtx.measureText(ev.ch);
      const asc = m.actualBoundingBoxAscent || ev.fontSize * 0.75;
      const desc = m.actualBoundingBoxDescent || ev.fontSize * 0.25;

      const isDesc = /[gjpqy]/.test(ev.ch);
      const isAsc = /[bdfhklt]/.test(ev.ch) || /[A-Z]/.test(ev.ch);

      // Create a pen path INSIDE the glyph box (more vertical motion)
      const x = ev.x;
      const w = Math.max(1, ev.w);
      const baseY = ev.baselineY;

      // choose vertical bias
      const yTop = baseY - asc * 0.70;
      const yMid = baseY + (isDesc ? desc * 0.75 : -asc * 0.10);
      const yBot = baseY + desc * 0.85;

      const y1 = isAsc ? yTop : (isDesc ? yBot : (rand(0,1) < 0.5 ? yTop : yBot));
      const y2 = yMid;
      const y3 = isDesc ? yBot : (isAsc ? yTop : yTop + (yBot - yTop) * 0.45);

      const points = [
        { x: x + w * 0.10, y: y1 },
        { x: x + w * 0.48, y: y2 },
        { x: x + w * 0.88, y: y3 },
      ];

      // Optional small jitter for "live pen"
      const jx = this.opts.penJitterX;
      const jy = this.opts.penJitterY;

      const t0 = performance.now();
      const duration = Math.max(30, ev.duration);
      const intensity = 0.75;

      await this.audio.penDown(intensity);

      // draw progressively along the polyline length
      while (true) {
        if (signal.aborted) break;
        while (this._paused && !signal.aborted) await new Promise(r => setTimeout(r, 50));
        if (signal.aborted) break;

        const now = performance.now();
        const p = clamp((now - t0) / duration, 0, 1);

        drawPartialPolyline(ctx, points, p, penW, jx, jy);

        // tip dab
        const tip = pointOnPolyline(points, p);
        ctx.beginPath();
        ctx.arc(tip.x + rand(-jx, jx), tip.y + rand(-jy, jy), penW * rand(0.45, 0.65), 0, Math.PI * 2);
        ctx.fill();

        composite();

        if (p >= 1) break;
        await nextFrame();
      }
    }

    _resolveFont(block) {
      const family = (block.fontFamily ?? this.opts.fontFamily);
      const size = this._resolveBlockFontSize(block);
      // Canvas expects CSS font shorthand order: style weight size family
      return `${this.opts.fontStyle} ${this.opts.fontWeight} ${size} ${family}`;
    }

    _resolveBlockFontSize(block) {
      const base = this.opts.fontSize;
      const size = block.fontSize ?? base;
      const scale = (block.size ?? 1);
      // allow clamp(...) strings etc
      if (typeof size === "string" && size.includes("clamp(")) return size;
      const px = parseCssPx(size);
      return `${px * scale}px`;
    }
  }

  function parseCssPx(v) {
    if (typeof v === "number") return v;
    const s = String(v || "").trim();
    if (s.startsWith("clamp(")) {
      // cannot resolve clamp to a number reliably; fall back to 44
      return 44;
    }
    const m = s.match(/([0-9.]+)\s*px/i);
    if (m) return parseFloat(m[1]);
    const n = parseFloat(s);
    return Number.isFinite(n) ? n : 44;
  }

  function nextFrame() {
    return new Promise(r => requestAnimationFrame(() => r()));
  }

  function sleep(ms, signal) {
    return new Promise((resolve, reject) => {
      if (signal?.aborted) return reject(Object.assign(new Error("aborted"), { name: "AbortError" }));
      const t = setTimeout(() => resolve(), ms);
      signal?.addEventListener?.("abort", () => { clearTimeout(t); reject(Object.assign(new Error("aborted"), { name: "AbortError" })); }, { once: true });
    });
  }

  function polylineLength(points) {
    let L = 0;
    for (let i = 1; i < points.length; i++) {
      const dx = points[i].x - points[i - 1].x;
      const dy = points[i].y - points[i - 1].y;
      L += Math.hypot(dx, dy);
    }
    return L;
  }

  function pointOnPolyline(points, t01) {
    const target = polylineLength(points) * clamp(t01, 0, 1);
    let acc = 0;
    for (let i = 1; i < points.length; i++) {
      const a = points[i - 1];
      const b = points[i];
      const seg = Math.hypot(b.x - a.x, b.y - a.y);
      if (acc + seg >= target) {
        const r = seg === 0 ? 0 : (target - acc) / seg;
        return { x: a.x + (b.x - a.x) * r, y: a.y + (b.y - a.y) * r };
      }
      acc += seg;
    }
    return points[points.length - 1];
  }

  function drawPartialPolyline(ctx, points, t01, width, jx, jy) {
    const target = polylineLength(points) * clamp(t01, 0, 1);
    let acc = 0;

    ctx.lineWidth = width * (0.92 + Math.random() * 0.18);

    ctx.beginPath();
    ctx.moveTo(points[0].x, points[0].y);

    for (let i = 1; i < points.length; i++) {
      const a = points[i - 1];
      const b = points[i];
      const seg = Math.hypot(b.x - a.x, b.y - a.y);

      if (acc + seg <= target) {
        ctx.lineTo(b.x + rand(-jx, jx), b.y + rand(-jy, jy));
        acc += seg;
      } else {
        const r = seg === 0 ? 0 : (target - acc) / seg;
        const x = a.x + (b.x - a.x) * r;
        const y = a.y + (b.y - a.y) * r;
        ctx.lineTo(x + rand(-jx, jx), y + rand(-jy, jy));
        break;
      }
    }

    ctx.stroke();
  }

  function normalizeOptions(o) {
    return {
      mode: o.mode || "replace",

      fontFamily: o.fontFamily || "cursive",
      fontSize: o.fontSize || "44px",
      fontWeight: o.fontWeight || "400",
      fontStyle: o.fontStyle || "normal",

      align: o.align || "center",
      verticalAlign: o.verticalAlign || "center",
      lineHeight: o.lineHeight ?? 1.22,
      inkColor: o.inkColor || "rgba(0,0,0,0.92)",

      // base timings
      minDelay: o.minDelay ?? 35,
      maxDelay: o.maxDelay ?? 95,
      timeScale: o.timeScale ?? 1.0,

      // letter draw duration
      revealMin: o.revealMin ?? 140,
      revealMax: o.revealMax ?? 260,

      // punctuation pause
      punctuationPauses: o.punctuationPauses ?? true,
      punctuationMin: o.punctuationMin ?? 180,
      punctuationMax: o.punctuationMax ?? 650,
      punctuationSet: new Set((o.punctuationChars || ".,;:!?").split("")),

      // extra pause at spaces (between words)
      wordPauseMin: o.wordPauseMin ?? 80,
      wordPauseMax: o.wordPauseMax ?? 220,

      // end-of-block pause
      linePauseMin: o.linePauseMin ?? 0,
      linePauseMax: o.linePauseMax ?? 0,

      // pen brush feel
      penWidth: o.penWidth ?? 3.2,
      penJitterX: o.penJitterX ?? 0.35,
      penJitterY: o.penJitterY ?? 0.45,

      // audio
      sound: {
        enabled: o.sound?.enabled ?? true,
        autoUnlock: o.sound?.autoUnlock ?? true,
        volume: o.sound?.volume ?? 0.16,
        roughness: o.sound?.roughness ?? 0.50,
        baseLevel: o.sound?.baseLevel ?? 0.08,
        transientLevel: o.sound?.transientLevel ?? 0.10,
        transientEvery: o.sound?.transientEvery ?? 4,
      },

      reducedMotion: o.reducedMotion ?? null,
      pauseWhenHidden: o.pauseWhenHidden ?? true,

      // If true, wait for web fonts (document.fonts) before first render.
      autoWaitForFonts: o.autoWaitForFonts ?? true,
    };
  }

  function create(container, options) {
    return new CanvasWriter(container, options);
  }

  return { create };
});
